<?php
declare(strict_types=1);
require_once __DIR__ . '/_common.php';

function auth_cfg(): array {
  return load_json(project_path('config/auth.json'));
}

function site_cfg(): array {
  return load_json(project_path('config/site.json'));
}

function is_logged_in(): bool {
  $cfg = auth_cfg();
  $cookie = $cfg['session_cookie_name'] ?? 'rcl_auth';
  if (!isset($_COOKIE[$cookie])) return false;
  $token = (string)$_COOKIE[$cookie];
  $expected = hash('sha256', ($cfg['salt'] ?? '') . ($cfg['password_sha256'] ?? '') . ($_SERVER['HTTP_USER_AGENT'] ?? ''));
  return hash_equals($expected, $token);
}

function require_login(): void {
  if (!is_logged_in()) {
    header('Location: login.php');
    exit;
  }
}

function login_attempt(string $user, string $pass): bool {
  $cfg = auth_cfg();
  if (($cfg['admin_user'] ?? 'admin') !== $user) return false;
  $salt = (string)($cfg['salt'] ?? '');
  $stored = (string)($cfg['password_sha256'] ?? '');
  $candidate = hash('sha256', $salt . $pass);
  return hash_equals($stored, $candidate);
}

function set_login_cookie(): void {
  $cfg = auth_cfg();
  $cookie = $cfg['session_cookie_name'] ?? 'rcl_auth';
  $token = hash('sha256', ($cfg['salt'] ?? '') . ($cfg['password_sha256'] ?? '') . ($_SERVER['HTTP_USER_AGENT'] ?? ''));
  setcookie($cookie, $token, [
    'expires' => time() + 3600 * 10,
    'path' => '/',
    'httponly' => true,
    'samesite' => 'Lax'
  ]);
}

function clear_login_cookie(): void {
  $cfg = auth_cfg();
  $cookie = $cfg['session_cookie_name'] ?? 'rcl_auth';
  setcookie($cookie, '', time() - 3600, '/');
}
