<?php
declare(strict_types=1);
require_once __DIR__ . '/_common.php';

function sessions_root(): string { return project_path('sessions'); }
function audit_path(): string { return project_path('audit/audit.ndjson'); }
function daily_hashes_path(): string { return project_path('audit/daily_hashes.csv'); }
function index_ndjson_path(): string { return project_path('indexes/sessions.ndjson'); }
function index_csv_path(): string { return project_path('indexes/sessions.csv'); }

function session_path(string $folder): string {
  $folder = trim($folder, "/\\");
  return sessions_root() . DIRECTORY_SEPARATOR . $folder;
}

function write_audit(string $action, string $folder, array $details = []): void {
  $site = load_json(project_path('config/site.json'));
  $tz = $site['timezone'] ?? 'UTC';
  $entry = [
    'ts' => now_iso($tz),
    'action' => $action,
    'folder' => $folder,
    'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
    'ua' => $_SERVER['HTTP_USER_AGENT'] ?? '',
    'details' => $details
  ];
  append_line(audit_path(), json_encode($entry, JSON_UNESCAPED_SLASHES));
}

function refresh_daily_hash_chain(): void {
  $auditFile = audit_path();
  if (!file_exists($auditFile)) return;

  $site = load_json(project_path('config/site.json'));
  $tz = $site['timezone'] ?? 'UTC';

  $lines = file($auditFile, FILE_IGNORE_NEW_LINES);
  $byDay = [];
  foreach ($lines as $ln) {
    $j = json_decode($ln, true);
    if (!is_array($j) || !isset($j['ts'])) continue;
    $dt = new DateTime($j['ts']);
    $dt->setTimezone(new DateTimeZone($tz));
    $day = $dt->format('Y-m-d');
    $byDay[$day][] = $ln;
  }

  ksort($byDay);
  $prev = '';
  $out = ["date,entries_sha256,prev_sha256,chain_sha256"];
  foreach ($byDay as $day => $dayLines) {
    $entriesBlob = implode("\n", $dayLines);
    $entriesHash = hash('sha256', $entriesBlob);
    $chain = hash('sha256', $entriesHash . $prev);
    $out[] = $day . "," . $entriesHash . "," . $prev . "," . $chain;
    $prev = $chain;
  }
  file_put_contents(daily_hashes_path(), implode(PHP_EOL, $out) . PHP_EOL, LOCK_EX);
}

function ensure_index_files_exist(): void {
  $csv = index_csv_path();
  if (!file_exists($csv)) {
    $hdr = ["date","case_number","hearing_type","site_code","room","folder"];
    file_put_contents($csv, implode(",", $hdr) . PHP_EOL, LOCK_EX);
  }
}

function append_index_row(array $row): void {
  ensure_index_files_exist();
  $csv = index_csv_path();
  $f = fopen($csv, 'a');
  if ($f) {
    fputcsv($f, $row);
    fclose($f);
  }
  append_line(index_ndjson_path(), json_encode($row, JSON_UNESCAPED_SLASHES));
}
