<?php
declare(strict_types=1);
require_once __DIR__ . '/_common.php';
require_once __DIR__ . '/_auth.php';
require_once __DIR__ . '/_layout.php';
require_once __DIR__ . '/_session_helpers.php';
require_login();

$site = site_cfg();
$fields = load_json(project_path('config/fields.json'));
$sessionFields = $fields['session_fields'] ?? [];
$tz = $site['timezone'] ?? 'UTC';

$error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $date = (string)($_POST['date'] ?? '');
  $time = (string)($_POST['time'] ?? '');
  $room = (string)($_POST['room'] ?? ($site['default_room'] ?? 'Room-1'));
  $siteCode = (string)($_POST['site_code'] ?? ($site['site_code'] ?? 'SITE'));
  $payload = [
    'meta' => [
      'created_at' => now_iso($tz),
      'site_name' => $site['site_name'] ?? '',
      'site_code' => $siteCode,
      'room' => $room,
    ],
    'session' => [
      'date' => $date,
      'start_time' => $time,
      'end_time' => '',
      'staff_operator' => (string)($_POST['staff_operator'] ?? ''),
    ],
    'fields' => []
  ];

  foreach ($sessionFields as $f) {
    $key = (string)($f['key'] ?? '');
    if ($key === '') continue;
    $val = (string)($_POST[$key] ?? '');
    if (($f['required'] ?? false) && $val === '') {
      $error = "Missing required field: " . ($f['label'] ?? $key);
      break;
    }
    $payload['fields'][$key] = $val;
  }

  if (!$error) {
    $caseNum = (string)($payload['fields']['case_number'] ?? 'NA');
    $hearingType = (string)($payload['fields']['hearing_type'] ?? 'NA');
    $folder = safe_slug($date . "_" . $time . "_" . $siteCode . "_" . $room . "_CASE-" . $caseNum);

    $spath = session_path($folder);
    if (is_dir($spath)) {
      $error = "Session folder already exists. Choose a different time.";
    } else {
      mkdir($spath . "/attachments", 0775, true);
      mkdir($spath . "/exports", 0775, true);

      save_json($spath . "/session.json", $payload);
      save_json($spath . "/participants.json", ['participants' => []]);
      save_json($spath . "/checklist.json", ['pre_session' => [], 'post_session' => []]);
      save_json($spath . "/incidents.json", ['incidents' => []]);

      append_index_row([
        'date' => $date,
        'case_number' => $caseNum,
        'hearing_type' => $hearingType,
        'site_code' => $siteCode,
        'room' => $room,
        'folder' => $folder
      ]);
      write_audit('create_session', $folder, ['case_number'=>$caseNum,'hearing_type'=>$hearingType]);
      refresh_daily_hash_chain();

      header("Location: view_session.php?folder=" . urlencode($folder));
      exit;
    }
  }
}

page_header('New Session');

if ($error) echo "<div class='card'><div class='pill' style='border-color:#b00020;color:#b00020'>" . h($error) . "</div></div>";

echo "<div class='card'>";
echo "<form method='post'>";
echo "<div class='grid'>";
echo "<div><label>Date</label><input type='date' name='date' required></div>";
echo "<div><label>Start Time</label><input type='time' name='time' required></div>";
echo "<div><label>Site Code</label><input name='site_code' value='" . h($site['site_code'] ?? 'SITE') . "' required></div>";
echo "<div><label>Room</label><input name='room' value='" . h($site['default_room'] ?? 'Room-1') . "' required></div>";
echo "<div><label>Staff Operator</label><input name='staff_operator' placeholder='Initials or name'></div>";
echo "<div></div>";
echo "</div><hr>";

foreach ($sessionFields as $f) {
  $key = (string)($f['key'] ?? '');
  $label = (string)($f['label'] ?? $key);
  $type = (string)($f['type'] ?? 'text');
  $req = (bool)($f['required'] ?? false);

  echo "<div style='margin:10px 0'>";
  echo "<label>" . h($label) . ($req ? " (required)" : "") . "</label>";
  if ($type === 'select') {
    echo "<select name='" . h($key) . "'" . ($req ? " required" : "") . ">";
    $opts = $f['options'] ?? [];
    echo "<option value=''>-- select --</option>";
    foreach ($opts as $o) {
      echo "<option value='" . h((string)$o) . "'>" . h((string)$o) . "</option>";
    }
    echo "</select>";
  } else {
    echo "<input name='" . h($key) . "'" . ($req ? " required" : "") . ">";
  }
  echo "</div>";
}

echo "<br><button class='btn primary' type='submit'>Create Session</button>";
echo "</form></div>";

page_footer();
